<?php

namespace App\Http\Controllers;

use App\Models\UserSkill;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;
use Illuminate\Support\Facades\Log;

class UserSkillController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            Log::info('Retrieving all user skills');
            $skills = UserSkill::with('user')->get();
            Log::info('Successfully retrieved user skills', ['count' => $skills->count()]);
            return response()->json(['success' => true, 'data' => $skills, 'message' => 'Skills retrieved successfully'], 200);
        } catch (Exception $e) {
            Log::error('Failed to retrieve user skills', ['error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Failed to retrieve skills: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            Log::info('Attempting to create new user skills', ['user_id' => $request->input('User_id')]);
            $validatedData = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Skills' => 'required|array',
                'Skills.*.Skill_name' => 'required|string|max:100',
                'Skills.*.Proficiency_level' => 'required|in:beginner,intermediate,advanced,expert',
            ]);

            $userId = $validatedData['User_id'];
            $skillsData = $validatedData['Skills'];

            // Delete existing skills for the user
            UserSkill::where('User_id', $userId)->delete();
            Log::info('Deleted existing skills for user', ['user_id' => $userId]);

            $skills = [];
            foreach ($skillsData as $skillData) {
                $skill = UserSkill::create([
                    'User_id' => $userId,
                    'Skill_name' => $skillData['Skill_name'],
                    'Proficiency_level' => $skillData['Proficiency_level'],
                ]);
                $skill->load('user');
                $skills[] = $skill;
            }

            Log::info('Successfully created user skills', ['user_id' => $userId, 'skill_count' => count($skills)]);
            return response()->json(['success' => true, 'data' => $skills, 'message' => 'Skills created successfully'], 201);
        } catch (Exception $e) {
            Log::error('Failed to create user skills', ['error' => $e->getMessage(), 'user_id' => $request->input('User_id')]);
            return response()->json(['success' => false, 'message' => 'Failed to create skills: ' . $e->getMessage()], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            Log::info('Retrieving user skill', ['skill_id' => $id]);
            $skill = UserSkill::with('user')->findOrFail($id);
            Log::info('Successfully retrieved user skill', ['skill_id' => $id, 'skill_name' => $skill->Skill_name]);
            return response()->json(['success' => true, 'data' => $skill, 'message' => 'Skill retrieved successfully'], 200);
        } catch (Exception $e) {
            Log::error('Failed to retrieve user skill', ['skill_id' => $id, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Skill not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            Log::info('Attempting to update user skill', ['skill_id' => $id]);
            $skill = UserSkill::findOrFail($id);
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Skill_name' => 'required|string|max:100',
                'Proficiency_level' => 'required|in:beginner,intermediate,advanced,expert',
            ]);

            $skill->update($validated);
            $skill->load('user');
            Log::info('Successfully updated user skill', ['skill_id' => $id, 'skill_name' => $skill->Skill_name]);
            return response()->json(['success' => true, 'data' => $skill, 'message' => 'Skill updated successfully'], 200);
        } catch (Exception $e) {
            Log::error('Failed to update user skill', ['skill_id' => $id, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Failed to update skill: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            Log::info('Attempting to delete user skill', ['skill_id' => $id]);
            $skill = UserSkill::with('user')->findOrFail($id);
            $skillName = $skill->Skill_name;
            $skill->delete();
            Log::info('Successfully deleted user skill', ['skill_id' => $id, 'skill_name' => $skillName]);
            return response()->json(['success' => true, 'data' => null, 'message' => "Skill '$skillName' has been deleted successfully"], 200);
        } catch (Exception $e) {
            Log::error('Failed to delete user skill', ['skill_id' => $id, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Failed to delete skill: ' . $e->getMessage()], 500);
        }
    }
}
